// Copyright Thomas Dreibholz, uzs6pq@ibm.rhrz.uni-bonn.de
// Lsung des Hamilton-Problems mittels Backtracking:
// Gegeben: Orte und Verbindungen
// Gesucht: Alle mgliche Hamilton'schen Kreise
// Aufruf: hamilton
//         Skript-Datei      = Dateiname
//         +f                = falsche Anstze zeigen

#include <stdio.h>
#include <stdlib.h>
#include <iostream.h>
#include <fstream.h>
#include <string.h>

class Ort;

// ===== Klasse "Weg" ====================================================
const int cMaxOrte = 350;

// Klasse "Weg"
// Diese Klasse dient dazu, einen Lsungsansatz bzw. eine Lsung
// des Wegeproblems als Tupel auszugeben. Wegen der Rekursion beim
// Backtracking ist ein Last-In-First-Out-Mechanismus notwendig.
class Weg
{
   public:
   Weg();                         // Konstruktor
   ~Weg();                        // Destruktor
   void Push(int nummer);         // Ortsnummer ablegen
   void Pop();                    // Ortsnummer wegnehmen
   int  Orte();                   // Anzahl der Orte auf dem Weg

   friend ostream& operator << (ostream& os, Weg* weg)   // Ausgabe

   private:
   int Zaehler;
   int Liste[cMaxOrte];
};

// Konstruktor fr Klasse "Weg"
Weg::Weg()
{
   Zaehler = 0;
}

// Destruktor fr Klasse "Weg"
Weg::~Weg()
{
}

// Neue Ortsnummer hinzufgen
void Weg::Push(int nummer)
{
   if(Zaehler < cMaxOrte - 1) {
      Liste[Zaehler++] = nummer;
   }
   else {
      cout << "Fehler bei Weg::Push()";
      exit(1);
   }
}

// Ortsnummer entfernen
void Weg::Pop()
{
   Zaehler--;
   if(Zaehler < 0) {
      cout << "Fehler bei Weg::Pop()";
      exit(1);
   }
}

// Ausgabe eines Weges als n-Tupel
ostream& operator <<(ostream& os, Weg* weg)
{
   os << "(";
   for(int i = 0;i < weg->Zaehler;i++) {
      os << weg->Liste[i];
      if(i != weg->Zaehler - 1) {
         os << ",";
      }
   }
   os << ")";
   return os;
}

// Anzahl der Orte auf dem Weg ermitteln
inline int Weg::Orte()
{
   return(Zaehler);
}


// ===== Klasse "Verbindung" =============================================

// Klasse "Verbindung"
// Diese Klasse dient zur Speicherung der Verbindung zwischen zwei Orten.
class Verbindung
{
   public:
   Verbindung(Ort* ziel);         // Konstruktor
   ~Verbindung();                 // Destruktor
   Ort* Zielort();                // Zielort der Verbindung

   // Verkettung von Verbindungen
   void        Naechste(Verbindung* verbindung);   // Zeiger setzen
   Verbindung* Naechste();                         // Zeiger abfragen

   private:
   Ort*        Ziel;               // Zielort
   Verbindung* NaechsteVerbindung; // Naechste Verbindung
};

// Konstruktor fr Klasse "Verbindung"
Verbindung::Verbindung(Ort* ziel)
{
   NaechsteVerbindung = NULL;
   Ziel               = ziel;
}

// Destruktor fr Klasse "Verbindung"
Verbindung::~Verbindung()
{
}

// Zielort ermitteln
inline Ort* Verbindung::Zielort()
{
   return(Ziel);
}

// Zeiger aus nchste Verbindung abfragen
inline Verbindung* Verbindung::Naechste()
{
   return(NaechsteVerbindung);
}

// Zeiger aus nchste Verbindung setzen
inline void Verbindung::Naechste(Verbindung* verbindung)
{
   NaechsteVerbindung = verbindung;
}


// ===== Klasse "Ort" ====================================================
const int cNichtBesucht = 1;      // Ort wurde noch nicht besucht
const int cBesucht      = 0;      // Ort wurde bereits besucht

// Klasse "Ort"
// Diese Klasse dient zur Speicherung der Orte und deren Verbindungen.
class Ort
{
   public:
   Ort(int nummer);                  // Konstruktor
   ~Ort();                           // Destruktor

   void Besucht(int besucht);        // Besucht-Flag ndern
   int  Besucht();                   // Wurde Ort schon besucht?
   int  Nummer();                    // Ortsnummer abfragen
   void Verbinde(Ort* ziel);         // Verbinde Ort mit "ziel"
   int  IstVerbundenMit(Ort* ziel);  // Abfrage, ob mit "ziel" verbunden

   // Verkettung von Orten
   void        Naechster(Ort* naechster);  // Nchsten Ort setzen
   Ort*        Naechster();                // Nchsten Ort abfragen
   Verbindung* Verbindungsliste();         // Zeiger auf Verbindungsliste

   private:
   int         WurdeBesucht;
   int         Ortsnummer;
   Ort*        NaechsterOrt;
   Verbindung* ErsteVerbindung;
};

// Konstruktor fr Klasse "Ort"
Ort::Ort(int nummer)
{
   Ortsnummer      = nummer;
   WurdeBesucht    = cNichtBesucht;
   NaechsterOrt    = NULL;
   ErsteVerbindung = NULL;
}

// Desstruktor fr Klasse "Ort"
Ort::~Ort()
{
   Verbindung* naechste = ErsteVerbindung;
   Verbindung* verbindung;

   while(naechste != NULL) {
      verbindung = naechste;
      naechste = naechste->Naechste();
      delete verbindung;
   }
}

// Besucht-Flag setzen oder lschen
inline void Ort::Besucht(int besucht)
{
   WurdeBesucht = besucht;
}

// Besucht-Flag abfragen
inline int Ort::Besucht()
{
   return(WurdeBesucht);
}

// Ortsnummer abfragen
inline int Ort::Nummer()
{
   return(Ortsnummer);
}

// Zeiger auf nchsten Ort abfragen
inline Ort* Ort::Naechster()
{
   return(NaechsterOrt);
}

// Nchsten Ort setzen
inline void Ort::Naechster(Ort* naechster)
{
   NaechsterOrt = naechster;
}

// Zeiger auf erste Verbindung
inline Verbindung* Ort::Verbindungsliste()
{
   return(ErsteVerbindung);
}

// Ort mit anderem verbinden
void Ort::Verbinde(Ort* ziel)
{
   Verbindung* verbindung = new Verbindung(ziel);
   if(verbindung == NULL) {
      cout << "Zu wenig Speicher!" << endl;
      exit(1);
   }

   verbindung->Naechste(ErsteVerbindung);
   ErsteVerbindung = verbindung;
}

// berprfen, ob eine Verbindung existiert
int Ort::IstVerbundenMit(Ort* ziel)
{
   Verbindung* verbindung = ErsteVerbindung;

   while(verbindung != NULL) {
      // Wenn eine Verbindung zum Startort existiert, ist die Lsung
      // gefunden.
      if(verbindung->Zielort()->Nummer() == ziel->Nummer()) {
         return(1);
      }
      verbindung = verbindung->Naechste();
   }
   return(0);
}

// ===== Klasse "Ortsliste" ==============================================

// Klasse "Ortsliste"
// Diese Klasse dient zur Speicherung und Verwaltung der Orte.
class Ortsliste
{
   public:
   Ortsliste();                   // Konstruktor
   ~Ortsliste();                  // Destruktor

   void  NeuerOrt(int nummer);    // Neuen Ort einfgen
   Ort  *FindeOrt(int nummer);    // Zeiger auf Ort ermitteln
   Ort  *ErsterOrt();             // Zeiger auf ersten Ort ermitteln
   void  VerbindeOrte(int n1, int n2);  // Orte verbinden
   int   Orte();                  // Anzahl der Orte in der Liste ermitteln

   private:
   Ort* Startort;                 // Zeiger auf ersten Ort
   int  Zaehler;                  // Zhler
}

// Konstruktor fr Klasse "Ortsliste"
Ortsliste::Ortsliste()
{
   Startort = NULL;
   Zaehler  = 0;
}

// Desstruktor fr Klasse "Ortsliste"
Ortsliste::~Ortsliste()
{
   Ort* naechster = Startort;
   Ort* ort;

   while(naechster != NULL) {
      ort = naechster;
      naechster = naechster->Naechster();
      delete ort;
   }
}

// Neuen Ort zur Ortsliste hinzufgen
void Ortsliste::NeuerOrt(int nummer)
{
   if(FindeOrt(nummer) != NULL) {
      cout << "Ort " << nummer << " existiert bereits." << endl;
   }

   Ort* ort = new Ort(nummer);
   if(ort == NULL) {
      cout << "Fehler bei Ortsliste::NeuerOrt()" << endl;
      exit(1);
   }
   ort->Naechster(Startort);
   Startort = ort;
   Zaehler++;
}

// Ort suchen
Ort* Ortsliste::FindeOrt(int nummer)
{
   Ort* ort = Startort;
   while(ort != NULL) {
      if(ort->Nummer() == nummer) {
         return(ort);
      }
      ort = ort->Naechster();
   }
   return(NULL);
}

// Ersten Ort ermitteln
inline Ort* Ortsliste::ErsterOrt()
{
   return(Startort);
}

// berprfen, ob zwei Orde bereits verbunden sind
int SindVerbunden(Ort* ort1, Ort* ort2)
{
   Verbindung* verbindung = ort1->Verbindungsliste();
   while(verbindung != NULL) {
      if(verbindung->Zielort() == ort2) {
         return(1);
      }
      verbindung = verbindung->Naechste();
   }
   return(0);
}

// Orte miteinander verbinden
void Ortsliste::VerbindeOrte(int n1, int n2)
{
   if(n1 == n2) {
      cout << "Ort " << n1 << " kann nicht mit sich selbst verbunden werden!" << endl;
      return;
   }
   Ort* ort1 = FindeOrt(n1);
   Ort* ort2 = FindeOrt(n2);
   if((ort1 == NULL) || (ort2 == NULL)) {
      cout << "Kann Orte " << n1 << " und " << n2 << " nicht verbinden:" << endl;
      cout << "Mindestens ein Ort existiert nicht!" << endl;
      exit(1);
   }

   if(!(SindVerbunden(ort1,ort2))) {
      ort1->Verbinde(ort2);
      ort2->Verbinde(ort1);
   }
   else {
      cout << "Die Orte " << n1 << " und " << n2 << " sind bereits verbunden!" << endl;
   }
}

// Anzahl der Orte in der Liste ermitteln
inline int Ortsliste::Orte()
{
   return(Zaehler);
}


// ===== Hauptteil========================================================


int        zeigeFalscheAnsaetze = 0;
Ortsliste* ortsliste = NULL;
Weg*       weg       = NULL;

// Statistik
int anzahlLoesungen =  0;   // Anzahl der Lsungen
int anzahlFalsch    =  0;   // Anzahl der falschen Anstze
int ersteLoesung    = -1;   // Anzahl der Schritte zur ersten Lsung

// Ortsliste und Verbindungen erstellen
void ErstelleOrtsliste(char* name)
{
   ifstream is(name);
   if(is.bad()) {
      cout << "Fehler beim ffnen der Datei <" << name << ">." << endl;
      exit(1);
   }

   ortsliste = new Ortsliste;
   weg       = new Weg;

   while(!is.eof()) {
      char befehl[80];
      int  a,b;

      is >> befehl;
      if(!(strcmp(befehl,"ort"))) {
         is >> a;
         cout << "Neuer Ort: " << a << endl;
         ortsliste->NeuerOrt(a);
      }
      else if(!(strcmp(befehl,"weg"))) {
         is >> a;
         is >> b;
         cout << "Verbinde Ort " << a << " mit " << b << endl;
         ortsliste->VerbindeOrte(a,b);
      }
      else {
         switch(befehl[0]) {
            case '#':
            case '/':
            case 0x00:
             is.getline(befehl,78);
             break;
            default:
               cout << "Syntaxfehler in Skriptdatei" << endl;
               cout << "Zeile <" << befehl << ">" << endl;
               cout << "Abbruch!" << endl;
               exit(1);
             break;
         }
      }
   }
   cout << endl;
}

// Ortsliste wieder lschen
void EntferneOrtsliste()
{
   delete weg;
   delete ortsliste;
}

// Der eigentliche Backtracking-Lsungsalgorithmus fr das Wegeproblem
void Backtracking(Ort* start, Ort* ort)
{
   // Zeiger auf erste Verbindung holen, den Ort in die Wegliste aufnehmen
   // und als besucht erklren
   Verbindung* verbindung = ort->Verbindungsliste();
   weg->Push(ort->Nummer());
   ort->Besucht(cBesucht);

   // Wenn der Weg soviele Orte enthlt, wie in der Ortsliste vorhanden
   // sind, dann mu von Ort "ort" eine direkte Verbindung zum Startort
   // existieren, damit eine Lsung existiert.
   if( (ortsliste->Orte() == weg->Orte()) && (ort->IstVerbundenMit(start)) ) {
      weg->Push(start->Nummer());
      cout << "** Lsung gefunden: " << weg << " **" << endl;
      weg->Pop();
      anzahlLoesungen++;
      if(ersteLoesung == -1) {
         ersteLoesung = anzahlFalsch;
      }
   }
   // Andernfalls mssen noch Orte besucht werden.
   else {
      // Durchlauf aller Verbindungen des aktuellen Ortes
      while(verbindung != NULL) {
         Ort* ziel = verbindung->Zielort();
         // Wenn der Zielort noch nicht besucht wurde, wird eine neue
         // Rekursionsstufe begonnen.
         if(ziel->Besucht() == cNichtBesucht) {
            Backtracking(start,ziel);
         }
         // Andernfalls ist der gemachte Lsungsansatz falsch und es wird
         // in der vorherigen Rekursionsstufe ber die nchste Verbindung
         // versucht, eine Lsung zu erhalten.
         else {
            if(zeigeFalscheAnsaetze == 1) {
               weg->Push(verbindung->Zielort()->Nummer());
               cout << "Kein Lsungsansatz: " << weg << endl;
               weg->Pop();
            }
            anzahlFalsch++;
         }
         verbindung = verbindung->Naechste();
      }
   }

   // Die Rekursionsstufe ist beendet. Jetzt wird der Ort wieder aus der
   // Wegliste entfernt und als nicht besucht markiert.
   weg->Pop();
   ort->Besucht(cNichtBesucht);
}


// Hauptprogramm
void main(int argc, char *argv[])
{
   cout << "Backtracking-Beispielprogramm #4 - Hamilton'sche Kreise" << endl;
   if(argc >= 2) {
      ErstelleOrtsliste(argv[1]);
      if(argc >= 3) {
         if(!(strcmp(argv[2],"+f"))) {
            zeigeFalscheAnsaetze = 1;
         }
      }
   }
   else {
      cout << "Aufruf: " << argv[0] << " [Skript] {+f}" << endl;
      exit(0);
   }

   // Alle Orte durchlaufen; dabei wird beginnend bei jedem Ort als
   // Startort nach Hamilton'schen Kreisen gesucht.
   Ort* ort = ortsliste->ErsterOrt();
   while(ort != NULL) {
      Backtracking(ort,ort);
      ort = ort->Naechster();
   }

   // Speicher freigeben
   EntferneOrtsliste();

   cout << "Keine (weiteren) Lsungen gefunden." << endl << endl;
   cout << "STATISTIK:" << endl;
   if(ersteLoesung != -1) {
      cout << "Erste Lsung nach " << ersteLoesung << " Fehlversuchen" << endl;
   }
   cout << "Anzahl der gefundenen Lsungen: " << anzahlLoesungen << endl;
   cout << "Anzahl der falschen Anstze:    " << anzahlFalsch << endl;
   cout << "GESAMT:                         " << anzahlLoesungen + anzahlFalsch << endl;
   char c = getc(stdin);
}
