// Copyright Thomas Dreibholz, uzs6pq@ibm.rhrz.uni-bonn.de
// Lsung des Labyrinth-Problems mittels Backtracking
// Aufruf: labyrint
//         Labyrinth-Datei     = Dateiname
//         Pausenzeit          = Pausenzeit in 1/1000 Sekunden
//         +s                  = Sackgassen anzeigen
//         -w                  = Nicht auf Tastendrcke warten
//         -sound              = Soundausgabe aus

#include <stdio.h>
#include <stdlib.h>
#include <iostream.h>
#include <fstream.h>
#include <string.h>
#include <graphics.h>
#include <conio.h>
#include <dos.h>

void GrafikAus();

// ===== Klasse "Labyrinth" ==============================================

// Konstanten
const int cBreite = 40;
const int cHoehe  = 25;

const char cMauer       = BROWN;
const char cWeg         = LIGHTGRAY;
const char cUndefiniert = DARKGRAY;
const char cStart       = LIGHTCYAN;
const char cZiel        = LIGHTRED;
const char cRoboter     = YELLOW;
const char cWegBenutzt  = LIGHTBLUE;

// Klasse "Labyrinth"
// In dieser Klasse wird das Labyrinth verwaltet
class Labyrinth
{
   public:
   Labyrinth(const char *name);      // Konstruktor
   ~Labyrinth();                     // Destruktor

   void Zeichne();                             // Labyrinth zeichnen
   int  Benutzbar(int offsetX, int offsetY);   // Feld benutzbar?
   void Benutze(int offsetX, int offsetY);     // Roboter auf Feld setzen

   int  XPosition();                  // X-Position abfragen
   int  YPosition();                  // Y-Position abfragen
   void Markiere(int x, int y);       // Position als benutzt markieren
   void Demarkiere(int x, int y);     // Position als unbenutzt demarkieren

   int Angekommen();                  // Abfrage, ob Ziel erreicht

   private:
   void ZeichneFeld(int x, int y);    // Feld zeichnen

   int  X,Y;
   int  StartX,StartY;
   int  ZielX,ZielY;
   char Feld[cBreite][cHoehe];
   char Benutzt[cBreite][cHoehe];
};

// Labyrinth-Konstruktor
Labyrinth::Labyrinth(const char *name)
{
   int x,y;

   // Initialisieren
   for(x = 0;x < cBreite;x++) {
      for(y = 0;y < cHoehe;y++) {
	 Feld[x][y]    = cUndefiniert;
         Benutzt[x][y] = 0;
      }
   }
   StartX = StartY = -1;
   ZielX  = ZielY  = -1;

   // Labyrinth einlesen
   ifstream is(name);
   if(is.bad()) {
      cout << "Fehler beim ffnen der Datei <" << name << ">." << endl;
      exit(1);
   }

   y = 0;
   while(!is.eof()) {
      char zeile[100];

      is.getline(zeile,98);
      while(strlen(zeile) < cBreite) {
         strcat(zeile,"~");
      }
      if(zeile[0] != '#') {
         if(y < cHoehe) {
            for(x = 0;x < cBreite;x++) {
               switch(zeile[x]) {
                  case '.':
                     Feld[x][y] = cWeg;
                   break;
                  case '*':
                     Feld[x][y] = cMauer;
                    break;
                  case 'S':
                     if(StartX == -1) {
                        Feld[x][y] = cWeg;
                        StartX = x;
                        StartY = y;
                     }
                     else {
                        cout << "Es wurden mindestens 2 Startorte definiert!" << endl << "Abbruch!" << endl;
                        exit(1);
                     }
                    break;
                  case 'Z':
                     if(ZielX == -1) {
                        Feld[x][y] = cWeg;
                        ZielX = x;
                        ZielY = y;
                     }
                     else {
                        cout << "Es wurden mindestens 2 Zielorte definiert!" << endl << "Abbruch!" << endl;
                        exit(1);
                     }
                    break;
               }
            }
            y++;
         }
      }
   }

   // Startort prfen
   if(StartX == -1) {
       cout << "Es wurde kein Startort definiert!" << endl << "Abbruch!" << endl;
       exit(1);
   }
   X = StartX;
   Y = StartY;

   // Zielort prfen
   if(ZielX == -1) {
       cout << "Es wurde kein Zielort definiert!" << endl << "Abbruch!" << endl;
       exit(1);
   }
}

// Labyrinth-Destruktor
Labyrinth::~Labyrinth()
{
}

// Feld zeichnen
void Labyrinth::ZeichneFeld(int x, int y)
{
   // Daten ermitteln
   int x1  = x * (640 / cBreite);
   int y1  = y * (480 / cHoehe);
   int x2  = ((x + 1) * (640 / cBreite)) - 1;
   int y2  = ((y + 1) * (480 / cHoehe)) - 1;
   int typ = Feld[x][y];
   if((x == StartX) && (y == StartY)) {
      typ = cStart;
   }
   if((x == ZielX) && (y == ZielY)) {
      typ = cZiel;
   }
   if((x == X) && (y == Y)) {
      typ = cRoboter;
   }
   if((typ == cWeg) && (Benutzt[x][y])) {
      typ = cWegBenutzt;
   }

   // Feld zeichnen
   setcolor(typ);
   rectangle(x1,y1,x2,y2);
   setfillstyle(SOLID_FILL,typ);
   floodfill(x1 + 2,y1 + 2,typ);
   if(typ == cRoboter) {
      setcolor(BLUE);
      line(x1,y1,x2,y2);
      line(x2,y1,x1,y2);
   }
}

// Labyrinth zeichnen
void Labyrinth::Zeichne()
{
   for(int x = 0;x < cBreite;x++) {
      for(int y = 0;y < cHoehe;y++) {
         ZeichneFeld(x,y);
      }
   }
   ZeichneFeld(X,Y);
}

// Feld benutzbar?
int Labyrinth::Benutzbar(int offsetX, int offsetY)
{
   int x = X + offsetX;
   int y = Y + offsetY;

   // Grenzen testen
   if((x < 0) || (x >= cBreite)) return(0);
   if((y < 0) || (y >= cHoehe))  return(0);

   // Benutzbar, wenn Feld unbenutzter Weg
   if((Feld[x][y] == cWeg) && (!Benutzt[x][y])) {
      return(1);
   }
   return(0);
}

// Roboter auf Feld bewegen
// Wichtig:
// Es findet keine Prfung statt, ob das Feld auch  wirklich benutzt werden
// darf! Dies vereinfacht die Rekursion beim Backtracking, da das aktuelle
// Feld nach dem rekursiven Aufruf von Backtracking() noch als belegt
// markiert ist, wenn der Roboter wieder auf dieses Feld gesetzt wird.
void Labyrinth::Benutze(int offsetX, int offsetY)
{
    X = X + offsetX;
    Y = Y + offsetY;
    ZeichneFeld(X - offsetX,Y - offsetY);
    ZeichneFeld(X,Y);
}

// X-Position abfragen
inline int Labyrinth::XPosition()
{
   return(X);
}

// Y-Position abfragen
inline int Labyrinth::YPosition()
{
   return(Y);
}

// Ziel erreicht?
int Labyrinth::Angekommen()
{
   if((X == ZielX) && (Y == ZielY)) {
      return(1);
   }
   return(0);
}

// Position als benutzt markieren
void Labyrinth::Markiere(int x, int y)
{
   if(Feld[x][y] == cWeg) {
      Benutzt[x][y] = 1;
      ZeichneFeld(x,y);
   }
   else {
      GrafikAus();
      cout << "Fehler bei Labyrinth::Markiere()!" << endl;
      exit(1);
   }
}

// Position als unbenutzt markieren
void Labyrinth::Demarkiere(int x, int y)
{
   if(Benutzt[x][y] == 1) {
      Benutzt[x][y] = 0;
      ZeichneFeld(x,y);
   }
   else {
      GrafikAus();
      cout << "Fehler bei Labyrinth::Demarkiere()!" << endl;
      exit(1);
   }
}


// ===== Hauptteil =======================================================


unsigned int wartezeit = 15;   // Wartezeit zwischen einzelnen Schritten
int warte              = 1;    // Warten bei gefundener Lsung
int zeigeSackgassen    = 0;    // Warten bei Tastendruck
int soundEin           = 1;    // Sound eingeschaltet

// Statistik
int anzahlLoesungen  =  0;
int anzahlSackgassen =  0;
int ersteLoesung     = -1;


// Grafikmodus einschalten
void GrafikEin()
{
   int gdriver = VGA, gmode = VGAHI, errorcode;

   initgraph(&gdriver, &gmode, "");
   errorcode = graphresult();
   if(errorcode != grOk) {
      cout << "Dieses Programm bentigt VGA-Karte und Treiber EGAVGA.BGI!" << endl;
      exit(1);
   }
}

// Grafikmodus ausschalten
void GrafikAus()
{
   getch();
   closegraph();
}


// Ton ausgeben
void Sound(int freq, int dauer)
{
   if(soundEin) {
      sound(freq);
      delay(dauer);
      nosound();
   }
}

// Pause
inline void Pause()
{
   delay(wartezeit);
}

// Lsung des Labyrinth-Problems mit Backtracking
void Backtracking(Labyrinth* labyrinth)
{
   // Aktuelle Position ermitteln
   int x = labyrinth->XPosition();
   int y = labyrinth->YPosition();
   int sackgasse = 1;

   // Aktuelles Feld als belegt markieren
   labyrinth->Markiere(x,y);

   // berprfen, ob Ziel erreicht wurde
   if(labyrinth->Angekommen()) {
      anzahlLoesungen++;
      if(ersteLoesung == -1) {
         ersteLoesung = anzahlSackgassen;
      }
      Sound(1000,200);
      if(warte) {
         getch();
      }
   }
   else {
      // Versuche rechtes Feld
      if(labyrinth->Benutzbar(1,0)) {
         sackgasse = 0;
         labyrinth->Benutze(1,0);
         Pause();
         Backtracking(labyrinth);
         labyrinth->Benutze(-1,0);
         Pause();
      }
      // Versuche unteres Feld
      if(labyrinth->Benutzbar(0,1)) {
         sackgasse = 0;
         labyrinth->Benutze(0,1);
         Pause();
         Backtracking(labyrinth);
         labyrinth->Benutze(0,-1);
         Pause();
      }
      // Versuche linkes Feld
      if(labyrinth->Benutzbar(-1,0)) {
         sackgasse = 0;
         labyrinth->Benutze(-1,0);
         Pause();
         Backtracking(labyrinth);
         labyrinth->Benutze(1,0);
         Pause();
      }
      // Versuche oberes Feld
      if(labyrinth->Benutzbar(0,-1)) {
         sackgasse = 0;
         labyrinth->Benutze(0,-1);
         Pause();
         Backtracking(labyrinth);
         labyrinth->Benutze(0,1);
         Pause();
      }

      // Wenn die Position eine Sackgasse ist, Fehler zhlen
      if(sackgasse) {
         anzahlSackgassen++;
         Sound(500,100);
         if(zeigeSackgassen) {
            if(warte) {
               getch();
            }
         }
      }
   }

   // Aktuelles Feld als unbelegt markieren
   labyrinth->Demarkiere(x,y);
}


// Hauptprogramm
void main(int argc, char* argv[])
{
   if(argc < 2) {
      cout << "Aufruf: " << argv[0] << " [Skript-Datei] {Pause in 1/1000 Sekunden} {+s} {-w} {-sound}" << endl;
      exit(1);
   }
   for(int i = 2; i < argc;i++) {
      if(!(strcmp(argv[i],"+s"))) {
         zeigeSackgassen = 1;
      }
      else if(!(strcmp(argv[i],"-w"))) {
         warte = 0;
      }
      else if(!(strcmp(argv[i],"-sound"))) {
         soundEin = 0;
      }
      else {
         wartezeit = atoi(argv[i]);
         if(wartezeit > 1000) {
            wartezeit = 1000;
         }
      }
   }

   // Initialisierung
   Labyrinth* labyrinth = new Labyrinth(argv[1]);
   if(labyrinth == NULL) {
      cout << "Nicht genug freier Speicher!" << endl;
      exit(0);
   }
   GrafikEin();

   // Backtracking
   labyrinth->Zeichne();
   Backtracking(labyrinth);
   Sound(2000,500);

   // Programmende
   delete labyrinth;
   GrafikAus();
   cout << "Backtracking-Beispielprogramm #3 - Labyrinth" << endl  << endl;
   cout << "STATISTIK:" << endl;
   if(ersteLoesung != -1) {
      cout << "Erste Lsung nach " << ersteLoesung << " Fehlversuchen" << endl;
   }
   cout << "Anzahl von Lsungen:   " << anzahlLoesungen << endl;
   cout << "Anzahl von Sackgassen: " << anzahlSackgassen << endl;
   cout << "GESAMT:                " << anzahlLoesungen + anzahlSackgassen << endl;
   char c = getc(stdin);
}
